﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO;
using System.Security.Cryptography;
using System.Diagnostics;
using System.Security.Principal;
using System.Windows.Forms;
using System.Security.Permissions;
using System.Security;

namespace OGSNET
{
    using OGSNET.Plugin;

    /** \brief アカウントのデータ */
    [Serializable]
    public struct AccountData :
        IComparable,
        IComparable<AccountData>,
        IEquatable<AccountData>
    {
        public string PluginId; /**< プラグインのID */
        public string UserName; /**< ユーザー名 */
        public string Password; /**< パスワード */

        public AccountData(string pluginId, string userName, string password)
        {
            this.PluginId = pluginId;
            this.UserName = userName;
            this.Password = password;
        }

        public AccountData(string pluginId) :
            this(pluginId, null, null)
        {
        }

        public static explicit operator Account(AccountData account)
        {
            return new Account(account.UserName, account.Password);
        }

        public bool IsEnable {
            get { return this.UserName != null && this.Password != null; }
        }

        #region Operators and Overrides

        public static bool operator ==(AccountData self, AccountData other){
            return self.Equals(other);
        }

        public static bool operator !=(AccountData self, AccountData other){
            return !self.Equals(other);
        }

        public static bool operator <(AccountData self, AccountData other){
            return self.CompareTo(other) < 0;
        }

        public static bool operator <=(AccountData self, AccountData other)
        {
            return self.CompareTo(other) <= 0;
        }

        public static bool operator >(AccountData self, AccountData other)
        {
            return self.CompareTo(other) > 0;
        }

        public static bool operator >=(AccountData self, AccountData other){
            return self.CompareTo(other) >= 0;
        }

        public int CompareTo(AccountData other)
        {
            if (this.UserName == null)
            {
                if (other.UserName == null)
                {
                    return 0;
                }
                else
                {
                    return 1;
                }
            }

            if (this.UserName != other.UserName)
            {
                return this.UserName.CompareTo(other.UserName);
            }

            return this.PluginId.CompareTo(other.PluginId);
        }

        public int CompareTo(object other)
        {
            if (other is AccountData)
            {
                return this.CompareTo((AccountData)other);
            }

            throw new ArgumentException();
        }

        public bool Equals(AccountData other)
        {
            return this.CompareTo(other) == 0;
        }

        public override bool Equals(object obj)
        {
            if (obj is AccountData)
            {
                return this.Equals((AccountData)obj);
            }

            return false;
        }

        public override string ToString()
        {
            return this.UserName;
        }

        public override int GetHashCode()
        {
            return base.GetHashCode();
        }

        #endregion
    }

    /** \brief アカウントデータの管理クラス */
    public class AccountDataControl
    {
        /**
         * \brief アカウントデータを扱うクラス
         * \arg \c path データファイルのパス
         * \arg \c key データの暗号化キー
         */
        public AccountDataControl(string path, string key){
            this.Accounts = new List<AccountData>();
            this.Path = path;

            // 暗号キーを生成
            HashAlgorithm hash;

            try
            {
                hash = new SHA256Cng();
            }
            catch (PlatformNotSupportedException) // CNG は Vista 以降対応
            {
                hash = new SHA256Managed();
            }
            
            this.Key = hash.ComputeHash(Encoding.Unicode.GetBytes(key));
            
            hash.Dispose();
        }

        /** \brief データを読み込む */
        public void Load(){
            var crypto = this.GetCrypto();
            var serializer = new BinaryFormatter();
            Stream fileStream;

            try
            {
                fileStream = new FileStream(this.Path, FileMode.Open, FileAccess.Read);
            }
            catch (IOException)
            {
                this.Accounts.Clear();
                return;
            }

            // IV を読み込む
            byte [] iv = new byte[crypto.BlockSize / 8];

            fileStream.Read(iv, 0, crypto.BlockSize / 8);
            crypto.IV = iv;

            // データを読み込む
            var cryptoStream = new CryptoStream(fileStream, crypto.CreateDecryptor(), CryptoStreamMode.Read);

            this.Accounts = (List<AccountData>)serializer.Deserialize(cryptoStream);

            // 解放処理
            cryptoStream.Close();
            cryptoStream.Dispose();
            fileStream.Dispose();
            crypto.Dispose();
        }

        /** \brief データを保存する */
        public void Save()
        {
            var crypto = this.GetCrypto();

            // IVをランダムで生成
            crypto.GenerateIV();

            var serializer = new BinaryFormatter();
            var fileStream   = new FileStream(this.Path, FileMode.Create, FileAccess.Write, FileShare.None);
            var cryptoStream = new CryptoStream(fileStream, crypto.CreateEncryptor(), CryptoStreamMode.Write);

            // IV を書き込む
            fileStream.Write(crypto.IV, 0, crypto.BlockSize / 8);

            // データを書き込む
            serializer.Serialize(cryptoStream, this.Accounts);

            // 解放処理
            cryptoStream.Close();
            fileStream.Dispose();
            cryptoStream.Dispose();
            crypto.Dispose();
        }

        /** \brief 暗号化オブジェクトを取得 */
        private SymmetricAlgorithm GetCrypto()
        {
            var crypto = new AesCryptoServiceProvider();

            crypto.KeySize = 256;
            crypto.Key     = this.Key;

            return crypto;
        }

        /** \brief データ保存先のファイルのパス */
        public string Path
        {
            private set;
            get;
        }

        /** \brief 暗号キー */
        public byte[] Key
        {
            private set;
            get;
        }

        /** \brief アカウントのデータ一覧 */
        public List<AccountData> Accounts
        {
            set;
            get;
        }
    }

    /** \brief アカウントに関するメソッド */
    
    class AccountUtility
    {
        public static string GetCurrentUserSid()
        {
            var user = WindowsIdentity.GetCurrent();
            var sid = user.User;

            return sid.ToString();
        }

        /** \brief セーブファイルの保存先 */
        public static string GetSaveFolderPath()
        {
            var path = Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData);
            path += '\\' + Application.ProductName;

            lock(typeof(AccountUtility))
            {
                if (!Directory.Exists(path))
                {
                    Directory.CreateDirectory(path);
                }
            }

            return path;

        }
    }

}
